# Next.js on Fly.io

## Recommended Configuration

### Dockerfile (Standalone Output)

Next.js standalone output creates a minimal production build:

```dockerfile
FROM node:20-alpine AS base

# Install dependencies
FROM base AS deps
WORKDIR /app
COPY package.json package-lock.json* ./
RUN npm ci

# Build
FROM base AS builder
WORKDIR /app
COPY --from=deps /app/node_modules ./node_modules
COPY . .
RUN npm run build

# Production
FROM base AS runner
WORKDIR /app
ENV NODE_ENV=production
ENV PORT=3000

# Create non-root user
RUN addgroup --system --gid 1001 nodejs
RUN adduser --system --uid 1001 nextjs

# Copy standalone build
COPY --from=builder /app/public ./public
COPY --from=builder --chown=nextjs:nodejs /app/.next/standalone ./
COPY --from=builder --chown=nextjs:nodejs /app/.next/static ./.next/static

USER nextjs
EXPOSE 3000

CMD ["node", "server.js"]
```

### next.config.js

Enable standalone output:

```js
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: 'standalone',
}

module.exports = nextConfig
```

### fly.toml

```toml
app = "my-nextjs-app"
primary_region = "ord"

[build]
  dockerfile = "Dockerfile"

[env]
  PORT = "3000"

[http_service]
  internal_port = 3000
  force_https = true
  auto_stop_machines = "stop"
  auto_start_machines = true

  [http_service.concurrency]
    type = "requests"
    soft_limit = 200
    hard_limit = 250

[[http_service.checks]]
  grace_period = "30s"
  interval = "30s"
  method = "GET"
  path = "/"
  timeout = "10s"
```

## Common Issues

### Static Assets Not Loading

**Cause:** Static files not copied in Dockerfile.

**Fix:** Ensure this line is in Dockerfile:
```dockerfile
COPY --from=builder --chown=nextjs:nodejs /app/.next/static ./.next/static
```

### API Routes Failing

**Cause:** Environment variables not set on Fly.io.

**Fix:**
```bash
fly secrets set DATABASE_URL="..." NEXTAUTH_SECRET="..."
```

### Image Optimization Errors

**Cause:** Sharp not installed for production image optimization.

**Fix:** Add to Dockerfile deps stage:
```dockerfile
RUN npm ci && npm install sharp
```

Or disable image optimization:
```js
// next.config.js
module.exports = {
  output: 'standalone',
  images: {
    unoptimized: true,
  },
}
```

### Build Too Slow

**Tips:**
1. Use `.dockerignore` to exclude node_modules, .next, .git
2. Leverage build cache with multi-stage builds
3. Use `npm ci` instead of `npm install`

**.dockerignore:**
```
node_modules
.next
.git
*.md
```

## Environment Variables

Next.js has build-time vs runtime env vars:

- `NEXT_PUBLIC_*` — Bundled at build time, exposed to browser
- Others — Available at runtime only (server-side)

For runtime secrets:
```bash
fly secrets set DATABASE_URL="..." API_KEY="..."
```

For build-time public vars:
```toml
[build.args]
  NEXT_PUBLIC_API_URL = "https://api.example.com"
```
